﻿#using System.Collections.Immutable;
#using System.Data;
#using Tessa.Extensions.Default.Shared;
#using Tessa.Extensions.Default.Shared.Workflow.Wf;
#using Tessa.Roles;

// Свойство, определяющее, есть ли активные подзадачи или нет
public bool HasSubTasks => Process.SubProcessCount > 0;

// Метод для заполнения структуры ролей и важных для маршрута полей из данных завершаемого задания
public void FillNextRolesFromTask(CardTask task)
{
    var nextRoles = new List<object>();
    foreach(var row 
        in task.Card.Sections.GetOrAddTable("WfResolutionPerformers").Rows.OrderBy(x => x.Get<int>("Order")))
    {
        if (row.State == CardRowState.Deleted)
        {
            continue;   
        }
    
        nextRoles.Add(
            new Dictionary<string, object>
            {
                {
                    "Role",
                    new Dictionary<string, object>
                    {
                        { "ID", row.Get<Guid>("RoleID") },
                        { "Name", row.Get<string>("RoleName") },
                    }
                }
            });
    }   
    Signal.NextRoles = nextRoles;
    Signal.NeedControl = task.Card.DynamicEntries.WfResolutions.WithControl;
    Signal.SingleTask = !task.Card.DynamicEntries.WfResolutions.MassCreation;
    Signal.MajorPerformer =  task.Card.DynamicEntries.WfResolutions.MajorPerformer;
    Signal.RoleID = task.Card.DynamicEntries.WfResolutions.ControllerID ?? Session.User.ID;
    Signal.RoleName = task.Card.DynamicEntries.WfResolutions.ControllerName ?? Session.User.Name;
    Signal.Planned = task.Card.DynamicEntries.WfResolutions.Planned;
    Signal.DurationInDays = task.Card.DynamicEntries.WfResolutions.DurationInDays;
    Signal.Comment = task.Card.DynamicEntries.WfResolutions.Comment;
    Signal.KindCaption = task.Card.DynamicEntries.WfResolutions.KindCaption;
    Signal.AuthorID = task.Card.DynamicEntries.WfResolutions.AuthorID;
    Signal.AuthorName = task.Card.DynamicEntries.WfResolutions.AuthorName;
    
    Signal.ParentTaskRowID = this.SignalObject.As<WorkflowEngineTaskSignal>().TaskIDs?.FirstOrDefault();
}


// Метод для расчета результата задания при отзыве
public void FillRevokeResult(CardTask task)
{
    task.Result = LocalizeName("WfResolution_Result_Cancelled");
}

// Метод для расчета результата задания при завершении задания с вариантом Завершить
public void FillCompleteTaskResult(CardTask task)
{   
    Dictionary<string, object> resolutionFields = task.Card.Sections[WfHelper.ResolutionSection].RawFields;
    string comment = resolutionFields.Get<string>(WfHelper.ResolutionCommentField).NormalizeComment();
    
    task.Result = string.IsNullOrEmpty(comment)
        ? LocalizeName("WfResolution_Result_CompletedWithoutComment")
    : string.Format(LocalizeAndEscapeFormat("$WfResolution_Result_Completed"), comment);
}

// Метод для заполнения результата задания при завершении задания с вариантом отправить
public void FillSendToPerformerTaskResult(CardTask task)
{
    string result = null;
    CardRow[] performersRows = WfHelper.TryGetPerformers(task);
       if (performersRows == null)
       {
           return;
       }
       string performers = string.Join(
           "; ",
           performersRows
               .Select(x =>
                   Localize(
                       x.Get<string>(WfHelper.ResolutionPerformerRoleNameField))));
       Dictionary<string, object> resolutionFields = task.Card.Sections[WfHelper.ResolutionSection].RawFields;
       string comment = resolutionFields.Get<string>(WfHelper.ResolutionCommentField).NormalizeComment();
       bool massCreation = performersRows.Length > 1 && resolutionFields.Get<bool>(WfHelper.ResolutionMassCreationField);
       if (massCreation)
       {
           bool withControl = resolutionFields.Get<bool>(WfHelper.ResolutionWithControlField);
           if (withControl)
           {
               result = string.IsNullOrEmpty(comment)
                   ? string.Format(
                       LocalizeAndEscapeFormat("$WfResolution_Result_SentToPerformerWithMassCreationAndWithControlAndWithoutComment"),
                        performers)
                    : string.Format(
                        LocalizeAndEscapeFormat("$WfResolution_Result_SentToPerformerWithMassCreationAndWithControlAndWithComment"),
                        performers,
                        comment);
            }
            else
            {
                result = string.IsNullOrEmpty(comment)
                    ? string.Format(
                        LocalizeAndEscapeFormat("$WfResolution_Result_SentToPerformerWithMassCreationAndWithoutComment"),
                        performers)
                    : string.Format(
                        LocalizeAndEscapeFormat("$WfResolution_Result_SentToPerformerWithMassCreationAndWithComment"),
                        performers,
                        comment);
            }
       }
       else
       {
            bool withControl = resolutionFields.Get<bool>(WfHelper.ResolutionWithControlField);
            if (withControl)
            {
                result = string.IsNullOrEmpty(comment)
                    ? string.Format(
                        LocalizeAndEscapeFormat("$WfResolution_Result_SentToPerformerWithControlAndWithoutComment"),
                        performers)
                    : string.Format(
                        LocalizeAndEscapeFormat("$WfResolution_Result_SentToPerformerWithControlAndWithComment"),
                        performers,
                        comment);
            }
            else
            {
                result = string.IsNullOrEmpty(comment)
                    ? string.Format(
                        LocalizeAndEscapeFormat("$WfResolution_Result_SentToPerformerWithoutControlAndWithoutComment"),
                        performers)
                    : string.Format(
                        LocalizeAndEscapeFormat("$WfResolution_Result_SentToPerformerWithoutControlAndWithComment"),
                        performers,
                        comment);
            }
       }
       task.Result = result;
}

// Метод, который удаляет запись о проекте из истории заданий в случае, если была вызвана отмена и нет дочерних записей в истории
public async Task TryRemoveTaskHistoryAsync(CardTask task)
{
    var executor = Context.DbScope.Executor;

    await executor.ExecuteNonQueryAsync(
        Context.DbScope.BuilderFactory
            .DeleteFrom("TaskHistory")
            .Where().C("TaskHistory", "RowID").Equals().P("RowID")
                .And().NotExists(b => b
                    .Select().V(null)
                    .From("TaskHistory", "pt").NoLock()
                    .Where().C("pt", "ParentRowID").Equals().C("TaskHistory", "RowID"))
            .Build(),
        CancellationToken.None,
        executor.Parameter("RowID", task.RowID));
}

#region Test support members
#region Constants

/// <summary>
/// Строка в сообщении валидации соответствующая успешному выполнению теста.
/// </summary>
private const string PassedStr = "Passed";

/// <summary>
/// Строка в сообщении валидации соответствующая ошибке при выполнении теста.
/// </summary>
private const string FailedStr = "Failed";

/// <summary>
/// Имя ключа, по которому в <see cref="CardInfoStorageObject.Info"/> карточки, в которой запущен бизнес-процесс, содержится значение флага, показывающего, запущен процесс из тестов или нет. Значение типа: <see cref="bool"/>.
/// </summary>
private const string IsLaunchedInTestKey = "IsLaunchedInTest";

/// <summary>
/// Имя ключа, по которому в <see cref="CardInfoStorageObject.Info"/> карточки, в которой запущен бизнес-процесс, содержится метод инициализации бизнес-процесса при выполнении из тестов. Значение типа: <see cref="Func{T, TResult}"/>, где T - <see cref="WorkflowEngineCompiledBase"/>, TResult - <see cref="ValueTask"/>.
/// </summary>
private const string TestInitializerActionKey = "TestInitializerAction";

#endregion

#region Properties

/// <summary>
/// Возвращает значение, показывающее, что процесс запущен из тестов.
/// </summary>
protected bool IsLaunchedInTest => this.ProcessHash.TryGet<bool>(IsLaunchedInTestKey);

#endregion

#region Protected Methods

/// <summary>
/// Добавляет в результаты валидации сообщение содержащее текст с признаком успешного выполнения теста "<see cref="PassedStr"/>[ <paramref name="suffix"/>]".
/// </summary>
/// <param name="suffix">Строка добавляемая к <see cref="PassedStr"/>.</param>
/// <remarks>Не выполняет действий, если процесс выполняется не из тестов.</remarks>
protected void Passed(string suffix = default)
{
    if (!this.IsLaunchedInTest)
    {
        return;
    }

    var str = PassedStr;

    if (!string.IsNullOrEmpty(suffix))
    {
        str += " " + suffix;
    }

    this.AddInfo(str);
}

/// <summary>
/// Добавляет в результаты валидации сообщение типа <see cref="ValidationResultType.Error"/> с сообщением об успешном выполнении.
/// </summary>
/// <remarks>
/// Созданное сообщение предназначено для остановки выполнения бизнес-процесса. Для этого при проверке результатов выполнения необходимо в методе <see cref="T:Tessa.Test.Default.Shared.Workflow.WeAssert.Passed"/> разрешить наличие ошибок в результате валидации.<para/>
/// Не выполняет действий, если процесс выполняется не из тестов.
/// </remarks>
protected void PassedWithStop()
{
    if (!this.IsLaunchedInTest)
    {
        return;
    }

    this.AddError(PassedStr);
}

/// <summary>
/// Добавляет в результаты валидации сообщение содержащее текст с признаком ошибки в тесте "<see cref="FailedStr"/>[ <paramref name="suffix"/>]".
/// </summary>
/// <param name="suffix">Строка добавляемая к <see cref="FailedStr"/>.</param>
/// <remarks>Не выполняет действий, если процесс выполняется не из тестов.</remarks>
protected void Failed(string suffix = default)
{
    if (!this.IsLaunchedInTest)
    {
        return;
    }

    this.AddError(FailedStr + " " + suffix);
}

/// <summary>
/// Инициализирует бизнес-процесс при выполнении из тестов.
/// </summary>
/// <returns>Асинхронная задача.</returns>
/// <remarks>Не выполняет действий, если процесс выполняется не из тестов.</remarks>
protected async ValueTask InitProcessAsync()
{
    var card = this.StoreCardObject;

    if(card is null)
    {
        return;
    }

    var info = card.TryGetInfo();
    
    if (info != null
        && info.Remove(IsLaunchedInTestKey, out var isLaunchedInTestObj)
        && ((bool) isLaunchedInTestObj))
    {
        this.ProcessHash.Add(IsLaunchedInTestKey, BooleanBoxes.True);

        if (info.Remove(TestInitializerActionKey, out object testInitializerFuncAsyncObj))
        {
            if(testInitializerFuncAsyncObj != null)
            {
                var initFuncAsync = (Func<WorkflowEngineCompiledBase, ValueTask>) testInitializerFuncAsyncObj;
                await initFuncAsync(this);
            }
        }
    }
}

#endregion
#endregion
